/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   Analog Trigger Resource Framework API Header File
::   Copyright   :   (C)2005-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   None
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file Resource_Trigger.h
    \brief The \ref BEHAVIOUR_ANALOG_TRIGGER provides the framework with the capability to sample input resources when
           a specific trigger occurs. See \ref triggeredadcsample */

#ifndef __RESOURCE_TRIGGER_H
#define __RESOURCE_TRIGGER_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <resource.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/
/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid analogue
           input attributes in the S_TriggerAttributes and S_TriggerCreateAttributes data structures.

Each mask definition applies to a particular member of one of the data structures.
    \code
    ...
    S_TriggerCreateAttributes CreateObj;

    // Will use DynamicObj
    CreateObj.uValidAttributesMask = USE_ANALOG_TRIGGER_DYNAMIC_ON_CREATE;
    CreateObj.DynamicObj.uValidAttributesMask = USE_ANALOG_TRIGGER_CONDITION;
    CreateObj.DynamicObj.eResourceCondition = RES_ENABLED;
    // The rest of the data structure can remain uninitialised because the framework will ignore it \endcode */    
typedef enum
{
    USE_ANALOG_TRIGGER_CONDITION        = 0x01,    /*!< Selects S_TriggerAttributes::eTriggerCondition */
    USE_ANALOG_TRIGGER_DYNAMIC_ON_CREATE= 0x02,    /*!< Selects S_TriggerCreateAttributes::DynamicObj  */
    USE_ANALOG_TRIGGER_NUMSLOTS         = 0x04,    /*!< Selects S_TriggerCreateAttributes::uCreateNumSlots */
    USE_ANALOG_TRIGGER_INPUT            = 0x08,    /*!< Selects S_TriggerAttributes::eTriggerInputResource */
    USE_ANALOG_TRIGGER_EDGE             = 0x10,    /*!< Selects S_TriggerAttributes::eTriggerEdge      */
    USE_ANALOG_TRIGGER_NOTIFY           = 0x20,    /*!< Selects S_TriggerAttributes::pfTriggerNotify   */

/* IF THIS TYPE EXCEEDS 0x80 THEN ALTER THE TYPE OF THE uValidAttributesMask MEMBER ACCORDINGLY */

} E_TriggerAttributesMask;

typedef enum
{
    ANALOG_TRIGGER_FALLING = 0,  /* (Default) */
    ANALOG_TRIGGER_RISING, 
} E_TriggerEdge;



typedef void (*TriggerCbackFuncPtrType)(E_ModuleResource in_eTriggerResource);

/*! \brief This data structure describes all of the trigger resource's runtime configuration attributes.

    The attributes are altered through the use of SetResourceAttributesBEHAVIOUR_ANALOG_TRIGGER(). The data
    structure does not need to be completely filled inorder to be used. The \b uValidAttributesMask is a bit
    field member that is used to identify which attributes are valid. Each attribute is identified with a
    separate bit mask that is logic-ORed into the a mask when an attribute is to be used.
    \code
    ...
    S_TriggerAttributes TriggerObj;

    // Only going to set the condition member
    TriggerObj.uValidAttributesMask = USE_ANALOG_TRIGGER_CONDITION;  \endcode */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_ANALOG_TRIGGER_CONDITION] that are valid for this this instance of the data structure.
    The valid attributes are described by the \ref E_TriggerAttributesMask enumeration
    \code
    ...
    S_TriggerAttributes TriggerObj;

    // Set the condition member and Trigger resource
    TriggerObj.uValidAttributesMask = USE_ANALOG_TRIGGER_CONDITION | USE_ANALOG_TRIGGER_RESOURCE;  \endcode */

    uint1 uValidAttributesMask;
/*! Set to RES_ENABLED to allow conversions to be triggered on this trigger. */
    E_ResourceCond   eResourceCondition;
/*! Selects which trigger edge to use. \n 
    Select this attribute with the \ref USE_ANALOG_TRIGGER_EDGE mask.  \n
    This attribute is assumed to be \ref ANALOG_TRIGGER_FALLING, by default. */
    E_TriggerEdge    eTriggerEdge;
/*! Selects which trigger input resource.   \n
    Select this attribute with the \ref USE_ANALOG_TRIGGER_INPUT mask.  \n
    This attribute is assumed to be \ref RES_ETRIG1, by default. */
    E_ModuleResource eTriggerInputResource; 
/*! CURRENTLY UNIMPLEMENTED \n
    Selects a trigger callback / notification funtion. \n
    Select this attribute with the \ref USE_ANALOG_TRIGGER_NOTIFY mask. 
    This attribute is assumed to be \ref NULL, by default. */
    TriggerCbackFuncPtrType pfTriggerNotify;
} S_TriggerAttributes;

typedef S_TriggerAttributes const* S_TriggerAttributesPtr;

/*! \brief This data structure describes the creation attributes for a trigger resource

    Included within the structure is the \p DynamicObj which represents the initial states for the resource's
    dynamic attributes. This data structure is referenced when the resource is created via the
    CreateResourceBEHAVIOUR_ANALOG_TRIGGER() function and \ref USE_ANALOG_TRIGGER_DYNAMIC_ON_CREATE is included in the
    S_TriggerCreateAttributes::uValidAttributesMask member.
    
    \code
    ...
    S_TriggerCreateAttributes CreateObj;

    // Will use DynamicObj from create
    CreateObj.uValidAttributesMask = USE_ANALOG_TRIGGER_DYNAMIC_ON_CREATE;
    // Will use the condition attribute of the dynamic object
    CreateObj.DynamicObj.uValidAttributesMask = USE_ANALOG_TRIGGER_CONDITION;
    CreateObj.DynamicObj.eResourceCondition = RES_ENABLED;
    // The rest of the data structure can remain uninitialised because the framework will ignore it *\
    // Trigger resource is by default timebased ANALOG_TIME_QUEUE2_AB and created automatically \endcode */    
typedef struct
{
/*! Logic-OR the attributes [\ref USE_ANALOG_TRIGGER_DYNAMIC_ON_CREATE] that are valid for this this instance of the data structure.
    The valid attributes are described by the \ref E_TriggerAttributesMask enumeration

    \code
    ...
    S_TriggerCreateAttributes CreateObj;

    // Will use DynamicObj and the min frequency
    CreateObj.uValidAttributesMask = USE_ANALOG_TRIGGER_DYNAMIC_ON_CREATE;
    CreateObj.DynamicObj.uValidAttributesMask = USE_ANALOG_TRIGGER_CONDITION | USE_ANALOG_TRIGGER_RESOURCE | USE_NUMBER_SLOTS;
    CreateObj.DynamicObj.eResourceCondition = RES_DISABLED;
    CreateObj.DynamicObj.eTriggerResource = RES_ADCTRIG_A;
    CreateObj.DynamicObj.uSlots = 4;
    // The rest of the data structure can remain uninitialised because the framework will ignore it \endcode */    
    uint1 uValidAttributesMask;
    uint1 uCreateNumSlots;
/*! Initial values of the runtime attributes, which can be later altered through a call to
    SetResourceAttributesBEHAVIOUR_ANALOG_TRIGGER(). Select with the \ref USE_ANALOG_TRIGGER_DYNAMIC_ON_CREATE bit mask */
    S_TriggerAttributes DynamicObj;
} S_TriggerCreateAttributes;

typedef S_TriggerCreateAttributes const* S_TriggerCreateAttributesPtr;

#pragma pack()
/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/

NativeError_S CreateResourceBEHAVIOUR_ANALOG_TRIGGER(E_ModuleResource, S_TriggerCreateAttributes const*);
NativeError_S DestroyResourceBEHAVIOUR_ANALOG_TRIGGER(E_ModuleResource);
NativeError_S SetResourceAttributesBEHAVIOUR_ANALOG_TRIGGER(E_ModuleResource, S_TriggerAttributes const*);

#endif /* __RESOURCE_TRIGGER_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
